<?php

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

require dirname(__FILE__).'/libs/MaksekeskusPayments.php';

require dirname(__FILE__).'/classes/PaymentMethod.php';

class MakeCommercePayments extends PaymentModule
{
    const MODULE_VERSION = '1.0.5';
    const MODULE_NAME = 'makecommercepayments';

    const STATUS_NAME = 'OS_MK_BILLING';


    const TYPE_BANK = 'banklinks';
    const TYPE_CARD = 'cards';
    const TYPE_OTHER = 'other';
    const TYPE_PAYLATER = 'payLater';

    const TEST = 0;
    const LIVE = 1;

    const DISPLAY_ORDER_PAGE = 1;
    const DISPLAY_WIDGET = 2;
    const IMG_SIZE_S = 's';
    const IMG_SIZE_M = 'm';
    const IMG_SIZE_L = 'l';

    const CACHE_VALID_TIME = 3600;

    const COUNTRY_ALL = 'all';

    const LOG_DEBUG = 'makecommerce-debug.log';

    private $fields = array(
        'secret_key',
        'shop_id',
        'publishable_key',
        'secret_key_test',
        'shop_id_test',
        'publishable_key_test',
        'server',
        'methods_display',
        'methods_named',
        'credit_display',
        'prefill_form',
        'logo_size',
        'methods_order',
    );

    private $lang_fields = array(
        'widget_title'
    );

    private $types = array(
        self::TYPE_BANK,
        self::TYPE_OTHER,
        self::TYPE_CARD,
        self::TYPE_PAYLATER
    );

    private static $api = null;

    private $html = '';

    public function __construct()
    {
        $this->name = self::MODULE_NAME;
        $this->tab = 'payments_gateways';
        $this->version = self::MODULE_VERSION;
        $this->need_instance = 0;
        $this->author = 'MakeCommerce.net';
        $this->bootstrap = true;
        $this->controllers = array('payment', 'validation', 'confirmation');

        parent::__construct();
        // $this->install(); // refresh hooks
        $this->displayName = 'MakeCommerce Payments';
        $this->description = 'Payment Gateway by Maksekeskus AS';
    }

    public function installHooks() {
        if (
            !parent::install() OR
            !$this->registerHook('header') OR //
            !$this->registerHook('displayBackOfficeHeader') OR //
            !$this->registerHook('paymentOptions') OR //
            !$this->registerHook('displayPaymentReturn') OR //
            !$this->registerHook('displayHeader') OR //
            !$this->registerHook('displayAdminOrder') OR
            !$this->registerHook('displayAdminOrderMain') OR
            !$this->registerHook('actionFrontControllerSetMedia') OR
            !$this->createOrderState() OR
            !$this->installTitles()
        ) return false; else return true;
    }

    public function installHooksOld() {
        if (
            !parent::install() OR
            !$this->registerHook('header') OR
            !$this->registerHook('adminOrder') OR
            !$this->registerHook('backOfficeHeader') OR
            !$this->registerHook('paymentOptions') OR
            !$this->registerHook('displayPaymentReturn') OR
            !$this->registerHook('displayHeader') OR
            !$this->createOrderState() OR
            !$this->installTitles()
        ) return false; else return true;
    }

    public function install()
    {
        $initStatus = false;
        if ( version_compare(_PS_VERSION_, "1.7.7.0", '<') ) {
            $initStatus = $this->installHooksOld();
        } else {  // register hooks for newer
            $initStatus = $this->installHooks();
        }
        if (!$initStatus) {
            return false;
        } else {
            $this->createTable();
            return true;
        }
    }

    public function createTable()
    {
        if (!Db::getInstance()->execute('
                CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'makecommerce_refunds` (
                  `id_makecommerce_refund` int(10) unsigned NOT NULL AUTO_INCREMENT,
                  `id_order` int(10) unsigned NOT NULL,
                  `refund_amount` decimal(20,6)	 NOT NULL,
                  `refund_date` datetime NOT NULL,
                  PRIMARY KEY (`id_makecommerce_refund`)
                  ) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=UTF8;
                ')) {
            return false;
        }
    }

    public function installTitles()
    {
        $languages = Language::getLanguages(false);
        foreach ($languages as $lang)
            $this->installTitle((int)$lang['id_lang']);

        return true;
    }

    public function installTitle($id_lang)
    {
        $values[self::prefixed('widget_title')][(int)$id_lang] = 'Pay by banklink or card';
        Configuration::updateValue(self::prefixed('widget_title'), $values[self::prefixed('widget_title')]);
    }

    public function isUsingNewTranslationSystem()
    {
        return false;
    }

    public function getContent()
    {
        if (Tools::isSubmit('submit'.$this->name)) {
            foreach ($this->fields as $field) {
                $this->updateConfig($field, Tools::getValue($field));
            }

            foreach ($this->lang_fields as $param){
                $langvalues = array();
                foreach (Language::getLanguages(false) as $key => $language){
                    $langvalues[$language['id_lang']] = Tools::getValue($param.'_'.$language['id_lang']);
                }
                $this->updateConfig($param, $langvalues, true);
            }

            // delete payment cache upon update
            $this->clearPaymentCache();

            $this->html .= $this->displayConfirmation($this->l('Settings updated from MakeCommerce servers'));
        }

        $this->html.=$this->showHeader();
        $this->showForm();
        return $this->html;
    }

    private function showHeader()
    {
        $this->smarty->assign(
            array(
                'path' => $this->getPathUri()
            )
        );
        return $this->display(__FILE__, 'settings_header.tpl');
    }

    private function showForm()
    {
        $field_values = array();
        $default_lang = (int)Configuration::get('PS_LANG_DEFAULT');
        $helper = new HelperForm();

        $fields_form[0]['form'] = array(
            'legend' => array(
                'title' => $this->l('Settings'),
            ),
            'input' => array(
                array(
                    'type' => 'select',
                    'label' => $this->l('Server'),
                    'name' => 'server',
                    'desc'     => $this->l('For Test Server see more about: ').'<a href="https://maksekeskus.ee/en/for-developers/test-environment/">MakeCommerce Test environment</a>',
                    'lang' => false,
                    'options' => array(
                        'query' => array(
                            array('id'=> self::TEST, 'name' => $this->l('Test server')),
                            array('id'=> self::LIVE, 'name' => $this->l('Live server'))
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    )
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Shop id'),
                    'desc'     => $this->l('You get the Shop ID and API keys from our Merchant Portal after sign-up'),
                    'name' => 'shop_id',
                    'size' => 30,
                    'form_group_class' => 'live_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Secret key'),
                    'name' => 'secret_key',
                    'form_group_class' => 'live_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Publishable key'),
                    'name' =>  'publishable_key',
                    'form_group_class' => 'live_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Shop id'),
                    'name' => 'shop_id_test',
                    'form_group_class' => 'test_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Secret key'),
                    'name' => 'secret_key_test',
                    'form_group_class' => 'test_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Publishable key'),
                    'name' =>  'publishable_key_test',
                    'form_group_class' => 'test_settings',
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        $fields_form[2]['form'] = array(
            'legend' => array(
                'title' => $this->l('Design'),
            ),
            'input' => array(
                array(
                    'type' => 'select',
                    'label' => $this->l('Display payment channels as'),
                    'name' => 'methods_display',
                    'required' => false,
                    'options' => array(
                        'query' => array(
                            array(
                                'id'=> self::DISPLAY_WIDGET,
                                'name' => $this->l('Widget')
                            ),
                            array(
                                'id'=> self::DISPLAY_ORDER_PAGE,
                                'name' => $this->l('List')
                            ),
                        ),
                        'id' => 'id',
                        'name' => 'name',
                    ),
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Widget title'),
                    'name' =>  'widget_title',
                    'form_group_class' => 'widget_setting',
                    'lang' => true,
                    'required' => false,
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Size of channel logos'),
                    'form_group_class' => 'widget_setting',
                    'name' => 'logo_size',
                    'required' => false,
                    'options' => array(
                        'query' => array(
                            array(
                                'id'=> self::IMG_SIZE_S,
                                'name' => $this->l('Small'),
                            ),
                            array(
                                'id'=> self::IMG_SIZE_M,
                                'name' => $this->l('Medium'),
                            ),
                            array(
                                'id'=> self::IMG_SIZE_L,
                                'name' => $this->l('Large'),
                            ),
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    )
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Credit cards in separate group'),
                    'form_group_class' => 'widget_setting',
                    'name' => 'credit_display',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'separate_group_on',
                            'value' => 1,
                            'label' => $this->l('Yes'),
                        ),
                        array(
                            'id' => 'separate_group_off',
                            'value' => 0,
                            'label' => $this->l('No'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Show payment method names')."*",
                    'form_group_class' => 'widget_setting',
                    'name' => 'methods_named',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'methods_named_on',
                            'value' => 1,
                            'label' => $this->l('Yes'),
                        ),
                        array(
                            'id' => 'methods_named_off',
                            'value' => 0,
                            'label' => $this->l('No'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Prefill credit card form with customer data'),
                    'name' => 'prefill_form',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'prefill_on',
                            'value' => 1,
                            'label' => $this->l('Yes')
                        ),
                        array(
                            'id' => 'prefill_off',
                            'value' => 0,
                            'label' => $this->l('No'),
                        )
                    )
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Payment methods order'),
                    'desc'     => $this->l('i.e.:  lhv, seb, swedbank'),
                    'name' =>  'methods_order',
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        $return_countries = array();
        $countries = Country::getCountries($this->context->employee->id_lang);
        foreach ($countries as $country) {
            $return_countries[] = array('id'=> $country['iso_code'], 'name' => $country['country']);
        }

        $helper->module = $this;
        $helper->name_controller = $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        $helper->default_form_language = $default_lang;
        $helper->allow_employee_form_lang = $default_lang;

        $helper->languages = Language::getLanguages(false);
        foreach ($helper->languages as $k => $language) {
            $helper->languages[$k]['is_default'] = (int)($language['id_lang'] == $helper->default_form_language);
        }

        $helper->title = $this->displayName;
        $helper->show_toolbar = true;
        $helper->toolbar_scroll = true;
        $helper->submit_action = 'submit'.$this->name;
        $helper->toolbar_btn = array(
            'save' => array(
                'desc' => $this->l('Save'),
                'href' => AdminController::$currentIndex.'&configure='.$this->name.'&save'.$this->name.
                '&token='.Tools::getAdminTokenLite('AdminModules'),
            ),
            'back' => array(
                'href' => AdminController::$currentIndex.'&token='.Tools::getAdminTokenLite('AdminModules'),
                'desc' => $this->l('Back to list')
            )
        );

        foreach ($this->fields as $param) {
            $field_values[$param] = Configuration::get(self::prefixed($param));
        }
        foreach ($this->lang_fields as $param)
        {
            foreach ($helper->languages as $key => $language)
            {
                $field_values[$param][$language['id_lang']] =
                    Configuration::get(self::prefixed($param), $language['id_lang']);
            }
        }

        $helper->fields_value = $field_values;

        $this->html .= $helper->generateForm($fields_form);
    }

    public function getPaymentMethodValues($country = null, $type = null)
    {
        $payment_methods = $this->getPaymentMethods();

        if (empty($payment_methods)) {
            return array();
        }

        $values = array();

        foreach ($payment_methods as $payment_method) {
            $method = $this->MkObjectToMethod($payment_method);

            if ((null === $type || $method->type == $type) &&
                (null === $country || $method->country == $country ||
                    ($method->country == self::COUNTRY_ALL && $type === null)
                )
            ) {
                $values[] = $method;
            }
        }
        $this->processMethods($values);

        return $values;
    }

    private function getMailVars()
    {
        return array(
            '{bankwire_owner}' => Configuration::get('BANK_WIRE_OWNER'),
            '{bankwire_details}' => nl2br(Configuration::get('BANK_WIRE_DETAILS')),
            '{bankwire_address}' => nl2br(Configuration::get('BANK_WIRE_ADDRESS'))
        );
    }

    public function confirmOrder($method_display_name = null)
    {
        $mailVars = $this->getMailVars();

        $this->validateOrder(
            $this->context->cart->id,
            Configuration::get(self::STATUS_NAME),
            $this->context->cart->getOrderTotal(),
            $this->displayName . ' ' . $method_display_name,
            NULL,
            $mailVars,
            $this->context->cart->id_currency,
            false,
            $this->context->customer->secure_key
        );
        $this->context->cookie->currentOrder = $this->currentOrder;
        $this->context->cookie->oldCartId = $this->context->cart->id;

        return new Order($this->currentOrder);
    }

    public static function prefixed($key)
    {
        return Tools::strtoupper(self::MODULE_NAME.'_'.$key);
    }

    public static function getConfig($key)
    {
        return Configuration::get(self::prefixed($key));
    }

    public function updateConfig($key, $value, $allow_html = false)
    {
        return Configuration::updateValue(self::prefixed($key), $value, $allow_html);
    }

    /**
     * Gets the api based on server configuration option (test/live)
     * @return Maksekeskus|null
     */
    public function getApi()
    {
        if (self::getConfig('server') == self::TEST) {
            $shop_id = self::getConfig('shop_id_test');
            $publishable_key = self::getConfig('publishable_key_test');
            $secret_key = self::getConfig('secret_key_test');
        } else {
            $shop_id = self::getConfig('shop_id');
            $publishable_key = self::getConfig('publishable_key');
            $secret_key = self::getConfig('secret_key');
        }

        if (!empty($shop_id) && !empty($publishable_key) && !empty($secret_key)) {
            $logger = new FileLogger(AbstractLogger::DEBUG);
            $logger->setFilename(_PS_ROOT_DIR_ . '/var/logs/' . self::LOG_DEBUG);
            self::$api = new MaksekeskusPayments(
                $shop_id,
                $logger,
                $publishable_key,
                $secret_key,
                self::getConfig('server') == self::TEST
            );
        }

        return self::$api;
    }

    /**
     * Gets available payment methods for the shop, filtered by methods_order configuration
     * @param $cache
     * @return array|false|mixed
     */
    private function getPaymentMethods($cache = true)
    {
        $config_key = 'payment_methods';

        if ($cache) {
            $payment_methods_json = self::getConfig($config_key);
            if (!empty($payment_methods_json)) {
                $payment_methods_data = json_decode($payment_methods_json);
                if (
                    empty($payment_methods_data->updated) ||
                    $payment_methods_data->updated + self::CACHE_VALID_TIME < time()
                ) {
                    $payment_methods_data = null;
                }
            }
        }
        if (!$cache || empty($payment_methods_data) ) {
            $api = $this->getApi();
            if (!is_null($api)) {
                $params = array(
                    'platform' => 'prestashop '._PS_VERSION_,
                    'module' => 'MakeCommerce '.$this->version
                );
                $environment = array(
                    'environment' => json_encode($params)
                );
                try {
                    $payment_methods_data = $api->getShopConfig($environment);
                    $payment_methods_data = $payment_methods_data->paymentMethods;
                } catch (Exception $e) {
                    PrestaShopLogger::addLog(
                        $e->getMessage(),
                        1,
                        null,
                        'MakeCommerce'
                    );
                    return false;
                }
            }

            $methods_order = self::getConfig('methods_order');

            $methods_order = array_filter(explode(',', $methods_order));
            $methods_order = array_map('trim', $methods_order);
            $methods_order = array_flip($methods_order);
            $payment_methods = array();
            if (isset($payment_methods_data) && $payment_methods_data) {
                foreach ($this->types as $type) {
                    foreach ($payment_methods_data->{$type} as $key => $payment_method) {
                        $payment_method->type = $type;
                        if (property_exists($payment_method, 'url')) {
                            unset($payment_method->url);
                        }

                        if (!property_exists($payment_method, 'country')) {
                            $payment_method->country = self::COUNTRY_ALL;
                        }
                        if (!empty($methods_order)) {
                            if(!array_key_exists($payment_method->name, $methods_order))
                                $payment_methods_default[] = $payment_method;
                        }
                        if (empty($methods_order)) {
                            $payment_methods[] = $payment_method;
                        } elseif (isset($methods_order[$payment_method->name])) {
                            if (empty($payment_methods[$payment_method->name])) {
                                $payment_methods[$payment_method->name] = array();
                            }

                            $payment_methods[$payment_method->name][] = $payment_method;
                        }
                    }
                }
            }


            if (!empty($methods_order)) {
                $payment_methods = array_reduce($payment_methods, 'array_merge', array());
            }

            if (!empty($methods_order)){
                $payment_methods = json_decode(json_encode($payment_methods), true);
                usort($payment_methods, array($this,"paymentMethodOrder"));
                $payment_methods = json_decode(json_encode($payment_methods), false);
                $merged_payment_methods = array_merge($payment_methods, $payment_methods_default);
                $payment_methods = $merged_payment_methods;
            }

            if (!empty($payment_methods)) {
                $object = new stdClass();
                $object->updated = time();
                $object->payment_methods = $payment_methods;
                $this->updateConfig($config_key, json_encode($object));
            }
        } else {
            $payment_methods = $payment_methods_data->payment_methods;
        }

        return $payment_methods;

    }

    public function paymentMethodOrder($a, $b){

        $methods_order = self::getConfig('methods_order');
        $methods_order = array_filter(explode(',', $methods_order));

        $a = array_search($a["name"], $methods_order);
        $b = array_search($b["name"], $methods_order);

        if ($a === false && $b === false) {
            return 0;
        }
        else if ($a === false) {
            return 1;
        }
        else if ($b === false) {
            return -1;
        }
        else {
            return $a - $b;
        }
    }

    private function createOrderState()
    {
        $orderStateExist = false;
        $orderStateId = Configuration::get(self::STATUS_NAME);
        $description = $this->l('Awaiting maksekeskus payment');

        if (strlen($description) > 64) {
            $description = substr($description, 0, 64);
        }

        if ($orderStateId) {
            $orderState = new OrderState($orderStateId);
            if($orderState->id && !$orderState->deleted) {
                $orderStateExist = true;
            }
        } else {
            $query = 'SELECT os.`id_order_state` '.
                'FROM `%1$sorder_state_lang` osl '.
                'LEFT JOIN `%1$sorder_state` os '.
                'ON osl.`id_order_state`=os.`id_order_state` '.
                'WHERE osl.`name`="%2$s" AND os.`deleted`=0';
            $orderStateId =  Db::getInstance()->getValue(sprintf($query, _DB_PREFIX_, $description));
            if ($orderStateId) {
                Configuration::updateValue(self::STATUS_NAME, $orderStateId);
                $orderStateExist = true;
            }
        }

        if (!$orderStateExist) {
            $languages = Language::getLanguages(false);
            $orderState = new OrderState();
            foreach ($languages as $lang) {
                $orderState->name[$lang['id_lang']] = $description;
            }

            $orderState->send_email = 0;
            $orderState->invoice = 0;
            $orderState->color = "lightblue";
            $orderState->unremovable = 0;
            $orderState->logable = 0;
            $orderState->delivery = 0;
            $orderState->hidden = 0;
            if($orderState->add()) {
                Configuration::updateValue(self::STATUS_NAME, $orderState->id);
                $orderStateExist = true;
            }
        }

        return $orderStateExist;
    }

    public function getImage($method)
    {
        $base_path = 'https://static.maksekeskus.ee/img/channel/lnd/';
        return  $base_path.$method.'.png';
    }

    public function getPaymentMethod($name, $payment_methods = array())
    {
        if (empty($name)) {
            return false;
        }

        if (empty($payment_methods)) {
            $payment_methods = $this->getPaymentMethodValues();
        }

        $result = false;
        foreach ($payment_methods as $method) {

            if ($method->code == $name) {
                $result = $method;
                break;
            }
        }

        return $result;
    }

    public function createPayment($transaction_id, $request_body){
        $api = $this->getApi();
        $api->createPayment($transaction_id, $request_body);
    }

    public function createTransaction(PaymentMethod $method, $country)
    {
        // added because refreshing on card payment gave an error
        if (!empty($this->context->cookie->currentOrder)) {
            $order = new Order($this->context->cookie->currentOrder);
            if ($order->getCurrentState() == Configuration::get('PS_OS_PAYMENT')) {
                $order = $this->confirmOrder($method->display_name);
            }
            // if cart id is different (meaning cart has changed) recreate the order
            elseif ($order->id_cart != $this->context->cart->id) {
                // set previous order state to cancelled
                if (Validate::isLoadedObject($order)) {
                    $order->setCurrentState(6);
                    $order->save();
                }
                $order = $this->confirmOrder($method->display_name);
            }
            // change payment name if payment method was changed
            if ($order->payment !== $this->displayName . ' ' . $method->display_name) {
                $order->payment = $this->displayName . ' ' . $method->display_name;
                $order->save();
            }
        } else {
            $order = $this->confirmOrder($method->display_name);
        }
        $api = $this->getApi();
        $transaction = null;

        if (!is_null($api) && Validate::isLoadedObject($order)) {
            $currency = new Currency($order->id_currency);
            $customer = new Customer($order->id_customer);
            $address = new Address($order->id_address_delivery);
            $val_url = $this->context->link->getModuleLink($this->name, 'val');

            if($country == 'all')
                $country = Country::getIsoById($address->id_country);

            $data = array(
                'transaction' => array(
                    'amount' => number_format(round(Tools::ps_round($order->total_paid, 2),2),2,".",""),
                    'currency' => $currency->iso_code,
                    'reference' => $order->id,
                    'transaction_url' => array(
                        'return_url' => array(
                            'url' => $val_url,
                            'method' => 'POST'
                        ),
                        'cancel_url' => array(
                            'url' => $val_url,
                            'method' => 'POST'
                        ),
                        'notification_url' => array(
                            'url' => $val_url,
                            'method' => 'POST'
                        )
                    )
                ),
                'customer' => array(
                    'email' => $customer->email,
                    'ip' => $this->getCustomerIp(),
                    'country' => $country,
                    'locale' => Language::getIsoById($order->id_lang)
                )
            );

            $transaction = $api->createTransaction($data);
        }

        return $transaction;
    }

    public function getTransaction(string $transaction_id)
    {
        $api = $this->getApi();

        return $api->getTransaction($transaction_id);
    }

    public function getBankUrlFromTransaction($transaction, $method)
    {
        $url = false;

        if (!is_null($transaction)) {
            foreach ($transaction->payment_methods->{MakeCommercePayments::TYPE_BANK} as $payment_method) {
                if ($payment_method->name == $method) {
                    $url = $payment_method->url;
                    break;
                }
            }
            foreach ($transaction->payment_methods->{MakeCommercePayments::TYPE_OTHER} as $payment_method) {
                if ($payment_method->name == $method) {
                    $url = $payment_method->url;
                    break;
                }
            }
            foreach ($transaction->payment_methods->{MakeCommercePayments::TYPE_PAYLATER} as $payment_method) {
                if ($payment_method->name == $method) {
                    $url = $payment_method->url;
                    break;
                }
            }
        }

        return $url;
    }

    public function getJsDataFromTrasaction($transaction)
    {
        $order = new Order((int)$transaction->reference);
        $customer = new Customer($order->id_customer);

        if (self::getConfig('server') == self::TEST) {
            $js_src = 'https://payment.test.maksekeskus.ee/checkout/dist/checkout.js';
            $publishable_key = self::getConfig('publishable_key_test');
        } else {
            $js_src = 'https://payment.maksekeskus.ee/checkout/dist/checkout.js';
            $publishable_key = self::getConfig('publishable_key');
        }

        $selectedMethod = Tools::getValue('method');
        $preselect = 'CC';
        if (stristr($selectedMethod, 'apple_pay')) {
            $preselect = 'AP';
        } elseif (stristr($selectedMethod, 'google_pay')) {
            $preselect = 'GP';
        }

        return array(
            'prefill' => self::getConfig('prefill_form'),
            'js_src' => $js_src,
            'publishable_key' => $publishable_key,
            'transaction_id' => $transaction->id,
            'currency' => $transaction->currency,
            'amount' => number_format(round($transaction->amount, 2),2,".",""), // number_format(round(Tools::ps_round($cart->getOrderTotal(true, 1), 2),2),2),
            'customer_email' => $transaction->customer->email,
            'customer_name' => $customer->firstname.' '.$customer->lastname,
            'shop_name' => $this->context->shop->name,
            'description' => sprintf($this->l('Order #%d'), $transaction->reference),
            'locale' => Language::getIsoById($order->id_lang),
            'quick_mode' => true,
            'completed' => 'makecommerce_cc_complete', // completed function name for CC payments
            'pre_select' => $preselect
        );
    }

    public function getCountries($payment_methods = null)
    {
        if (null === $payment_methods) {
            $payment_methods = $this->getPaymentMethods();
        }

        $countries = array();
        if(isset($payment_methods) && $payment_methods){
            foreach ($payment_methods as $payment_method) {
                if ($payment_method->country != self::COUNTRY_ALL && !isset($countries[$payment_method->country])) {
                    $countries[$payment_method->country] = 1;
                }
            }
        }
        return array_keys($countries);
    }

    public function appendPaymentMethods(&$types, $with_credit_cards = false)
    {
        $payment_methods = $this->getPaymentMethods();
        $grouped_methods = array();
        $cards = array();
        if (isset($payment_methods) && $payment_methods) {
            foreach ($payment_methods as $payment_method) {
                $method = $this->MkObjectToMethod($payment_method);
                $key = sprintf('%s_%s', $method->type, $method->country);
                if (!isset($grouped_methods[$key])) {
                    $grouped_methods[$key] = array();
                }
                $grouped_methods[$key][] = $method;

                if ($with_credit_cards && $method->type == self::TYPE_CARD) {
                    $cards[] = $method;
                }
            }
        }
        foreach ($types as &$type) {
            $key = $type->getKey();
            if (isset($grouped_methods[$key])) {
                $type->methods = array_merge($grouped_methods[$key], $cards);
                $this->processMethods($type->methods, self::IMG_SIZE_S);
            }
        }
    }

    public function hookHeader()
    {
        $this->context->controller->addJS($this->getPathUri().'views/js/makecommerce.js');
        $this->context->controller->addCSS($this->getPathUri().'views/css/makecommerce.css');
    }

    public function hookDisplayHeader()
    {
        return $this->hookHeader();
    }

    public function hookBackOfficeHeader()
    {
        if (version_compare(_PS_VERSION_, '9.0.0', '<')) {
            $this->context->controller->addJquery();
        }
        $this->context->controller->addJS($this->getPathUri().'views/js/makecommerce.js');
    }

    public function hookDisplayBackOfficeHeader()
    {
        return $this->hookBackOfficeHeader();
    }

    public  function hookDisplayAdminOrder($params)
    {
        return $this->hookDisplayAdminOrderMain($params);
    }

    public function hookDisplayPaymentReturn($params)
    {
        $order = $params['order'];

        if ($order->hasBeenPaid())
        {
            $status = 'ok';
        }
        else
        {
            $status = 'error';
        }

        $this->smarty->assign(array(
            'status' => $status,
            'link_to_order' => $this->getOrderConfUrl($order)
        ));
        return $this->display(__FILE__, 'payment_return.tpl');
    }

    public function hookPaymentOptions($params) {
        $methods_display = (int)self::getConfig('methods_display');
        $credit_cards_separately = (bool)self::getConfig('credit_display');
        $methods_named = (bool)self::getConfig('methods_named');
        $countries = $this->getCountries();
        $invoice_address = new Address((int)$this->context->cart->id_address_delivery);
        $current_country = Tools::strtolower(Country::getIsoById((int)$invoice_address->id_country));

        switch ($methods_display) {
            case self::DISPLAY_ORDER_PAGE:
                $payment_methods = $this->getPaymentMethodValues();
                if(count($countries)> 1){
                    $show_country_code = TRUE;
                }else{
                    $show_country_code = FALSE;
                }
                $this->context->smarty->assign(array(
                    'show_country_code' => $show_country_code,
                    'methods_named' => $methods_named
                ));
                break;

            case self::DISPLAY_WIDGET:
                $payment_methods = $this->getPaymentMethodValues();
                if(count($countries)> 1){
                    $show_country_code = TRUE;
                }else{
                    $show_country_code = FALSE;
                }
                $this->appendPaymentMethods($payment_methods, !$credit_cards_separately);
                $this->context->smarty->assign(array(
                    'widget_title' => trim(Configuration::get(
                        self::prefixed('widget_title'),
                        $this->context->language->id)),
                    'widget' => TRUE,
                    'show_country_code' => $show_country_code,
                    'separate_group' => $credit_cards_separately,
                    'methods_named' => $methods_named
                ));
                break;
        }

        if (empty($payment_methods)) {
            return;
        }
        $img_size = self::getConfig('logo_size');
        $this->processMethods($payment_methods, $img_size);
        $this->context->smarty->assign(array(
            'payment_methods' => $payment_methods,
            'logo_size' => $img_size,
            'mk_ajax_url' => $this->context->link->getModuleLink($this->name, 'ajax'),
            'order_total' => $this->context->cart->getOrderTotal(),
            'flag_url' => _MODULE_DIR_,
            'countries' => $countries,
            'current_country' => $current_country,
            'one_page_checkout' => (bool)Module::isEnabled('thecheckout')
        ));

        $newOption = new PaymentOption();
        $newOption->setModuleName($this->name)
            ->setCallToActionText($this->l('Pay by MakeCommerce'))
            ->setAction($this->context->link->getModuleLink($this->name, 'confirmation', array(), true))
            ->setAdditionalInformation($this->fetch('module:makecommercepayments/views/templates/hook/payments.tpl'));
        $payment_options = [
            $newOption,
        ];

        return $payment_options;
    }

    public function hookDisplayAdminOrderMain($params)
    {
        $error = '';
        $id_order = $params['id_order'];
        $order = new Order((int)$id_order);
        $maksekeskusPayment = FALSE;
        foreach ($order->getOrderPaymentCollection() as $payment) {
            if($payment->payment_method == 'MakeCommerce Payments') {
                $maksekeskusPayment = TRUE;
                break;
            }
        }
        if ($maksekeskusPayment) {

            if (Tools::isSubmit('submitMKRefund')) {
                $this->_doTotalRefund($id_order);
            }

            if (Tools::isSubmit('submitMKRefundPartial')) {
                $refund_amount = Tools::getValue('mk_partial_refund');
                $order_total = $order->total_paid;
                $refunded_total = $this->refundedAmount($id_order);
                $remaining_amount = $order_total - $refunded_total;

                if (isset($refund_amount) && $refund_amount > 0) {
                    if ($remaining_amount - $refund_amount >= 0) {
                        $this->_doPartialRefund($id_order, $refund_amount);
                    } else {
                        $error = $this->l('Refund amount greater than available to refund.');
                    }
                } else {
                    $error = $this->l('Invalid refund value.');
                }
            }

            if ($this->context->cookie->__isset('refund_status')) {
                // if ($this->context->cookie->__get('refund_status') != "")
                    $error = $this->context->cookie->__get('refund_status');
                $this->context->cookie->__unset('refund_status');
                $this->context->cookie->write();
            }
            $this->smarty->assign(array(
                'error' => $error,
                'id_order' => $id_order,
                'refunded' => $this->refundedAmount($id_order),
                'refund_details' => $this->getRefundDetails($id_order),
                'total_amount' => $order->total_paid
            ));
            return $this->display(_PS_MODULE_DIR_ .'makecommercepayments/makecommercepayments.php', 'adminOrder.tpl');
        }
    }

    protected function _doTotalRefund($id_order)
    {
        $order = new Order($id_order);
        $amount = number_format(round(Tools::ps_round($order->total_paid, 2),2),2,".","");
        $transaction_id = Db::getInstance()->getValue(
            'SELECT `transaction_id`
            FROM `'._DB_PREFIX_.'order_payment`
            WHERE order_reference = \''.$order->reference.'\''
        );

        $request_body = array(
            'amount' => $amount,
            'comment' => 'Order: '.$id_order.' refund'
        );

        $api = $this->getApi();
        try {
            $api->createRefund($transaction_id, $request_body);
            $this->saveRefundDetails($id_order, $amount);
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                    $e->getMessage(),
                    1,
                    null,
                    'MakeCommerce'
            );
            $this->context->controller->errors[] = $e->getMessage();

            if ($e instanceof MKPaymentsException) {
                $error_raw = json_decode($e->getRawContent(), true);
                $error_msg = $error_raw['message'];
            } else {
                $error_msg = $this->l('Total refund failed. See log for details.');
            }

            $this->context->cookie->__set('refund_status', $error_msg);
            $this->context->cookie->write();
            Tools::redirect($_SERVER['HTTP_REFERER']);
        }
        $order->setCurrentState(Configuration::get('PS_OS_REFUND'));
        Tools::redirect($_SERVER['HTTP_REFERER']);
    }

    protected function _doPartialRefund($id_order, $amount)
    {
        $order = new Order($id_order);
        $partial_amount = number_format(round(Tools::ps_round($amount, 2),2),2,".","");
        $transaction_id = Db::getInstance()->getValue(
            'SELECT `transaction_id`
            FROM `'._DB_PREFIX_.'order_payment`
            WHERE order_reference = \''.$order->reference.'\''
        );

        $request_body = array(
            'amount' => $partial_amount,
            'comment' => 'Order: '.$id_order.' refund'
        );
        $api = $this->getApi();
        try {
            $api->createRefund($transaction_id, $request_body);
            $this->saveRefundDetails($id_order, $partial_amount);
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                    $e->getMessage(),
                    1,
                    null,
                    'MakeCommerce'
            );
            if ($e instanceof MKPaymentsException) {
                $error_raw = json_decode($e->getRawContent(), true);
                $error_msg = $error_raw['message'];
            } else {
                $error_msg = $this->l('Total refund failed. See log for details.');
            }
            $this->context->controller->errors[] = $e->getMessage();
            $this->context->cookie->__set('refund_status', $error_msg);
            $this->context->cookie->write();
            Tools::redirect($_SERVER['HTTP_REFERER']);
        }
        $order->setCurrentState(Configuration::get('PS_OS_REFUND'));
        Tools::redirect($_SERVER['HTTP_REFERER']);
    }

    private function refundedAmount($id_order)
    {
        $refunded_total = Db::getInstance()->ExecuteS(
            'SELECT SUM(refund_amount) AS total FROM `'._DB_PREFIX_.'makecommerce_refunds` WHERE id_order = \''.$id_order.'\''
        );
        return $refunded_total[0]['total'];
    }

    private function saveRefundDetails($id_order, $amount)
    {
        Db::getInstance()->Execute(
            'INSERT INTO `'._DB_PREFIX_.'makecommerce_refunds` (`id_order`, `refund_amount`, `refund_date`)
                    VALUES('.(int)$id_order.', \''.$amount.'\', \''.date('Y-m-d H:i:s').'\')'
        );
    }

    private function getRefundDetails($id_order)
    {
        return Db::getInstance()->ExecuteS(
            'SELECT * FROM `'._DB_PREFIX_.'makecommerce_refunds` WHERE id_order = \''.$id_order.'\''
        );
    }

    protected function processMethods(array &$methods)
    {

        foreach ($methods as &$method)
        {
            $params = array(
                'method' => $method->code,
                'country' => $method->country
            );

            $method->link = $this->context->link->getModuleLink(
                $this->name,
                (empty($method->methods) ? 'confirmation' : 'payments'),
                $params
            );
            $method->img = $method->logo_url;
        }
    }

    public function MkObjectToMethod(stdClass $mk_method)
    {
        if (!(isset($mk_method->min_amount) && $mk_method->min_amount > 0))
            $mk_method->min_amount = 0;
        if (!(isset($mk_method->max_amount) && $mk_method->max_amount > 0))
            $mk_method->max_amount = 0;
        return new PaymentMethod(
            $mk_method->name,
            $mk_method->display_name,
            $mk_method->display_name,
            $mk_method->country,
            $mk_method->type,
            $mk_method->logo_url,
            $mk_method->min_amount,
            $mk_method->max_amount
        );
    }

    public function getCustomerIp()
    {
        if (
            !empty($_SERVER['HTTP_CF_CONNECTING_IP']) &&
            filter_var($_SERVER['HTTP_CF_CONNECTING_IP'], FILTER_VALIDATE_IP)
        ) {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        }

        return $_SERVER['REMOTE_ADDR'];
    }

    protected function clearPaymentCache()
    {
        $query = 'DELETE FROM %sconfiguration WHERE `name` LIKE \'%s%%\'';
        Db::getInstance()->execute(sprintf(
            $query,
            _DB_PREFIX_,
            self::prefixed('payment_methods')
        ));
    }

    public function getOrderConfUrl($order)
    {
        return $this->context->link->getPageLink(
            'order-confirmation',
            true,
            $order->id_lang
        );
    }

    /**
     * Checks whether product is in stock in sufficient amount and if out of stock product ordering is disabled
     * @param $id_cart
     * @return bool
     */
    public function validateCart($id_cart)
    {
        $valid = true;
        $cart = new Cart($id_cart);
        $products = $cart->getProducts();

        foreach ($products as $product) {
            if ($product['cart_quantity'] > $product['quantity_available'] && $product['out_of_stock'] != 1 && $product['allow_oosp'] != 1) {
                $valid = false;
            }
        }

        return $valid;
    }

    public function hookActionFrontControllerSetMedia($params)
    {
        if (Tools::getValue('submitReorder')) {
            self::restoreCart();
        }
    }

    public static function restoreCart()
    {
        $context = Context::getContext();
        $oldCartId = $context->cookie->oldCartId;
        if (!$context->cart->id && $oldCartId) {
            $oldCart = new Cart($oldCartId);
            $duplication = $oldCart->duplicate();

            if ($duplication && $duplication['success']) {
                $context->cookie->id_cart = $duplication['cart']->id;
                $context->cart = $duplication['cart'];
                CartRule::autoAddToCart($context);
            }

            unset($context->cookie->oldCartId);
        }
    }

    public static function redirect($url) {
        if (version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            Tools::redirect($url);
        } else {
            Tools::redirectLink($url);
        }
    }
}
